/*************************************************************************
*FILE: cypress_i2c.c
*SW-COMPONENT: cypress touch driver
*DESCRIPTION: c file for the i2c functionalities used by cypress driver.
*COPYRIGHT: (C) 2020 Robert Bosch GmbH. All Rights Reserved. Confidential.
* This software is licensed under the terms of the End User License
* Agreement (EULA) of Robert Bosch GmbH. Any use is subject
* to agreement and compliance with such EULA, as applicable.
***************************************************************************/

#include <string.h>
#include <malloc.h>
#include <stdint.h>
#include <errno.h>
#include <unistd.h>
#include "touch.h"
#include "cypress_i2c.h"

#ifndef UNITTEST
#include "i2c_client.h"
#include "input/mtouch_log.h"
#endif

int cypress_read_reg(int fd, uint16_t reg, uint16_t len, void* buf)
{
    uint8_t write_buf[1];
    int ret;
    bool retry = false;

    if (fd == -1)
        return -EINVAL;

    write_buf[0] = reg & 0xff;

retry_read:
    ret = i2c_combined_writeread(fd, write_buf, 1, buf, len);
    if (ret != len) {
        if (!retry) {
            if ((usleep(CYPRESS_CORE_WAKEUP_TIMEOUT * 1000)) == 0) {
                retry = true;
                goto retry_read;
            } else {
                ret = errno;
            }
        } else {
            ret = -EIO;
        }
    } else {
        ret = 0;
    }

    return ret;
}

int cypress_write_reg(int fd, uint16_t reg, uint16_t len, const void *val)
{
    uint8_t *buf;
    int count;
    int ret;
    bool retry = false;

    if (fd == -1)
        return -EINVAL;

    count = len + 1;
    buf =(uint8_t *) malloc(count);
    if (!buf)
        return -ENOMEM;

    buf[0] = reg & 0xff;
    memcpy(&buf[1], val, len);

retry_write:
    ret = i2c_write(fd, buf, count);
    if (ret != count) {
        if (!retry) {
            if ((usleep(CYPRESS_CORE_WAKEUP_TIMEOUT * 1000)) == 0) {
                retry = true;
                goto retry_write;
            } else {
                ret = errno;
            }
        } else {
            ret = -EIO;
        }
    } else {
        ret = 0;
    }

    free(buf);
    return ret;
}

int cypress_i2c_open(char* i2c_devname, unsigned slave_id, unsigned i2c_speed)
{
    int fd;

    fd = i2c_open(i2c_devname);
    if (fd == -1)
        return -ENODEV;

    if((i2c_set_slave_addr(fd, slave_id, I2C_ADDRFMT_7BIT)) == -1)
        return -1;

    if((i2c_set_bus_speed(fd, I2C_SPEED_HIGH, &i2c_speed)) == -1)
        return fd; /* default speed is set,even on failure */

    return fd;
}

void cypress_i2c_close(int32_t fd)
{
    if (fd != -1)
        i2c_close(fd);

    fd = 0;
}

int get_qc_libi2c_funcs(const char* libname, qc_i2c_funcs_t* funcs)
{
    if(0 == strcmp(libname, "qc-i2c-client")) {
        funcs->open = cypress_i2c_open;
        funcs->close = cypress_i2c_close;
        funcs->read_reg = cypress_read_reg;
        funcs->write_reg = cypress_write_reg;
        return 0;
    }

    errno = EINVAL;
    return -1;
}
